<?php

namespace App\Http\Controllers;

use App\Models\Stock;
use Illuminate\Http\Request;

class StockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $stocks = Stock::all()->load(['product', 'files']);

        return response()->json([
            'success' => true,
            'data' => $stocks
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $rules = [
            'stock' => 'nullable|int',
            'color_id' => 'nullable|exists:colors,id',
            'size_id' => 'nullable|exists:sizes,id',
            'product_id' => 'required|exists:products,id',
            'files' => 'nullable|array',
            'files.*' => 'nullable|numeric|exists:files,id',
        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $stock = Stock::create($data);

        if (isset($data['files'])) {
            $stock->files()->attach($data['files'], ['product_id' => $data['product_id']]);
        }

        return response()->json([
            'success' => true,
            'data' => $stock
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Stock $stock)
    {
        $stock->load('product');
        return response()->json([
            'success' => true,
            'data' => $stock
        ], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Stock $stock)
    {
        $rules = [
            'stock' => 'nullable|int',
            'color_id' => 'nullable|exists:colors,id',
            'size_id' => 'nullable|exists:sizes,id',
            'product_id' => 'required|exists:products,id',
            'files' => 'nullable|array',
            'files.*' => 'nullable|numeric|exists:files,id',
        ];
        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $stock->update($data);

        if (isset($data['files'])) {
            $stock->files()->attach($data['files'], ['product_id' => $data['product_id']]);
        }

        $stock->load(['files']);
        
        return response()->json([
            'success' => true,
            'data' => $stock
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Stock $stock)
    {
        $deleted = $stock->delete();
        $responseCode = $deleted ? 200 : 404;
        $message = $deleted ? 'Stock deleted' : 'Stock not found';
        $success = $deleted ? true : false;

        return response()->json(
            [
                'success' => $success,
                "message" => $message,
                'data' => $stock,
            ],
            $responseCode
        );
    }
}