<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {

        $rules = [
            "category_id" => "nullable|exists:categories,id",
            "subcategory_id" => "nullable|exists:subcategories,id",
            "search_term" => "nullable|string",
            "limit" => "nullable|integer",
            "offset" => "nullable|integer",
        ];

        $validator = Validator::make($request->input(), $rules);

        $data = $validator->validated();

        $query = Product::with(['category', 'subcategory', 'stocks', 'stocks.color','stocks.size', 'pricetypes', 'files']);

        if (isset($data['search_term'])) {
            $query->where(function ($query) use ($data) {
                $query->whereHas('category', function ($query) use ($data) {
                    $query->where('name', 'LIKE', '%' . $data['search_term'] . '%');
                });
                $query->orWhereHas('subcategory', function ($query) use ($data) {
                    $query->where('name', 'LIKE', '%' . $data['search_term'] . '%');
                });
                $query->orWhere('name', 'LIKE', '%' . $data['search_term'] . '%');
            });
        }


        if (isset($data['category_id'])) {
            $query->where('category_id', $data['category_id']);
        }

        if (isset($data['subcategory_id'])) {
            $query->where('subcategory_id', $data['subcategory_id']);
        }

        if (isset($data['limit'])) {
            $query->limit($data['limit']);
        }

        if (isset($data['offset'])) {
            $query->offset($data['offset'] * $data['limit']);
        }

        $products = $query->get();

        return response()->json([
            'success' => true,
            'data' => $products,
        ], 200);
    }

    public function indexPublic(Request $request): JsonResponse
    {
        $rules = [
            "category" => "nullable|exists:categories,slug",
            "last_added" => "nullable|in:true",
            "limit" => "nullable|integer",
            "offset" => "nullable|integer",
            "search_term" => "nullable|string",
            "subcategory" => "nullable|exists:subcategories,id",
            "color" => "nullable|exists:colors,id",
            "max_price" => "nullable|numeric",
            "is_top_product" => "nullable|in:true",
            "is_favorite" => "nullable|in:true,false",
        ];

        $query = Product::with(['category', 'subcategory', 'stocks', 'stocks.color','stocks.size', 'pricetypes', 'files']);

        $validator = Validator::make($request->input(), $rules);

        $params = $validator->validated();

        $query->where('show_in_shop', true);

        if (isset($params['category'])) {
            $query->whereHas('category', function ($query) use ($params) {
                $query->where('slug', $params['category']);
            });
        }

        if (isset($params['subcategory'])) {
            $query->where('subcategory_id', $params['subcategory']);
        }

        if (isset($params['last_added'])) {
            $query->orderBy('created_at', 'desc');
        }

        if (isset($params['search_term'])) {
            // Search by name or description
            $query->where(function ($query) use ($params) {
                $query->where(DB::raw('LOWER(name)'), 'LIKE', '%' . strtolower($params['search_term']) . '%')
                    ->orWhere(DB::raw('LOWER(description)'), 'LIKE', '%' . strtolower($params['search_term']) . '%');
            });
        }

        if (isset($params['color'])) {
            $query->whereHas('stocks', function ($query) use ($params) {
                $query->where('color_id', $params['color']);
            });
        }

        if (isset($params['max_price'])) {
            $query->whereHas('pricetypes', function ($query) use ($params) {
                $query->where('price', '<=', $params['max_price']);
                $query->where('name', 'Web');
            });
        }

        if (isset($params['is_top_product'])) {
            $is_top_product = filter_var($params['is_top_product'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
            $query->where('is_top_product', $is_top_product);
            $query->orderBy('updated_at', 'desc');
        }

        if (isset($params['limit'])) {
            $query->limit($params['limit']);
        }

        if (isset($params['offset'])) {
            $query->offset($params['offset']);
        }

        if (isset($params['is_favorite'])) {
            if (!Auth::check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not authenticated',
                ], 401);
            }
            $is_favorite = filter_var($params['is_favorite'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
            $query->whereHas('customersMarkedAsFavorite', function ($query) use ($is_favorite) {
                $customersIds = Auth::user()->customers->pluck('id')->toArray();
                $query->whereIn('customer_id', $customersIds);
            });
        }

        $products = $query->get();

        return response()->json([
            'success' => true,
            'data' => $products,
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $rules = [
            'name' => 'required|string',
            'category_id' => 'nullable|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'supplier_name' => 'required|string',
            'description' => 'nullable|string',
            'purchase_price' => 'nullable|numeric',
            'is_top_product' => 'nullable|boolean',
            'show_in_shop' => 'nullable|boolean',
            'pricelist' => 'nullable|array',
            'pricelist.*.price' => 'required|numeric',
            'pricelist.*.pricetype_id' => 'required|exists:pricetypes,id',

        ];


        $validator = Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();


        $product = Product::create([
            'name' => $data['name'],
            'purchase_price' => $data['purchase_price'],
            'category_id' => $data['category_id'],
            'subcategory_id' => $data['subcategory_id'],
            'supplier_name' => $data['supplier_name'],
            'description' => $data['description'] ?? null,
            'is_top_product' => $data['is_top_product'] ?? false,
            'show_in_shop' => $data['show_in_shop'] ?? true,
        ]);

        if (isset($data['pricelist'])) {
            $product->pricetypes()->attach($data['pricelist']);
        }

        $product->load(['category', 'subcategory', 'stocks', 'pricetypes']);

        return response()->json([
            'success' => true,
            'data' => $product,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Product $product): JsonResponse
    {
        $product->load(['category', 'subcategory', 'stocks.files', 'pricetypes']);
        return response()->json([
            'success' => true,
            'data' => $product,
        ]);
    }

    public function showPublic(Product $product): JsonResponse
    {
        $product->load(['category', 'subcategory', 'stocks.files', 'stocks.color', 'stocks.size', 'pricetypes', 'files']);
        return response()->json([
            'success' => true,
            'data' => $product,
        ]);
    }

    public function searchProduct(Request $request): JsonResponse
    {

        $rules = [
            'searchTerm' => 'required|string',
        ];

        $validator = Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $products = Product::where(DB::raw('LOWER(name)'), 'LIKE', '%' . strtolower($data['searchTerm']) . '%')
            ->get()
            ->load(['category', 'files', 'subcategory', 'stocks', 'stocks.color','stocks.size', 'pricetypes']);



        return response()->json([
            'success' => true,
            'data' => $products,
        ], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product): JsonResponse
    {
        $rules = [
            'name' => 'required|string',
            'category_id' => 'nullable|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'supplier_name' => 'required|string',
            'description' => 'nullable|string',
            'purchase_price' => 'nullable|numeric',
            'is_top_product' => 'nullable|boolean',
            'show_in_shop' => 'nullable|boolean',
            'pricelist' => 'nullable|array',
            'pricelist.*.price' => 'required|numeric',
            'pricelist.*.pricetype_id' => 'required|exists:pricetypes,id',
        ];

        $validator = Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $product->update([
            'name' => $data['name'],
            'purchase_price' => $data['purchase_price'],
            'category_id' => $data['category_id'],
            'subcategory_id' => $data['subcategory_id'],
            'supplier_name' => $data['supplier_name'],
            'description' => $data['description'] ?? $product->description,
            'is_top_product' => $data['is_top_product'] ?? $product->is_top_product,
            'show_in_shop' => $data['show_in_shop'] ?? $product->show_in_shop,
        ]);

        if (isset($data['pricelist'])) {
            $product->pricetypes()->detach();
            $product->pricetypes()->sync($data['pricelist']);
        }

        return response()->json([
            'success' => true,
            'data' => $product,
        ], 200);
    }

    public function markAsFavorite(Product $product): JsonResponse
    {

        $customers = Auth::user()->customers;
        $customersIds = $customers->pluck('id')->toArray();

        $isFavorite = $product->customersMarkedAsFavorite()->whereIn('customer_id', $customersIds)->exists();

        if ($isFavorite) {
            $product->customersMarkedAsFavorite()->detach($customersIds);
            $product->load(['category', 'files', 'subcategory', 'stocks', 'stocks.color','stocks.size', 'pricetypes']);

            return response()->json([
                'success' => true,
                'data' => $product,
                'message' => 'Product removed from favorites',
            ], 200);
        }

        $product->customersMarkedAsFavorite()->attach($customersIds);

        $product->load(['category', 'files', 'subcategory', 'stocks', 'stocks.color','stocks.size', 'pricetypes']);

        return response()->json([
            'success' => true,
            'data' => $product,
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product): JsonResponse
    {
        $deleted = $product->delete();
        $responseCode = $deleted ? 200 : 404;
        $message = $deleted ? 'Product deleted' : 'Product not found';
        $success = $deleted ? true : false;

        return response()->json(
            [
                'success' => $success,
                "message" => $message,
                'data' => $product,
            ],
            $responseCode
        );
    }
}
