<?php

namespace App\Http\Controllers;

use App\Models\Category;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CategoryController extends Controller
{

    public function index(): JsonResponse
    {
        $categories = Category::all()->load(['subcategories', 'file']);

        return response()->json([
            'success' => true,
            'data' => $categories,
        ], 200);
    }

    public function indexPublic(): JsonResponse
    {
        $categories = Category::all()->load(['subcategories', 'file']);

        return response()->json([
            'success' => true,
            'data' => $categories,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        //
        $rules = [
            'name' => 'required|string',
            'subcategories.*' => 'string',
            'file_id' => 'nullable|exists:files,id',
        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $category = Category::create([
            'name' => $data['name'],
            'file_id' => isset($data['file_id']) ? $data['file_id'] : null,
        ]);

        if (isset($data['subcategories'])) {
            foreach ($data['subcategories'] as $subcategory) {
                $category->subcategories()->create([
                    'name' => $subcategory,
                ]);
            }
        }

        $category->load('subcategories');

        return response()->json([
            'success' => true,
            'data' => $category,
        ], 201);

    }


    public function show(Category $category): JsonResponse
    {
        $category->load('subcategories', 'file');
        return response()->json([
            'success' => true,
            'data' => $category,
        ]);
    }

    public function showPublic(Category $category): JsonResponse
    {
        $category->load('subcategories', 'file');
        return response()->json([
            'success' => true,
            'data' => $category,
        ]);
    }

    public function update(Request $request, Category $category): JsonResponse
    {
        $rules = [
            'name' => 'required|string',
            'new_subcategories.*' => 'string',
            'file_id' => 'nullable|exists:files,id',
        ];

        $validator = \Validator::make($request->input(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 400);
        }

        $data = $validator->validated();

        $category->update([
            'name' => $data['name'],
            'file_id' => isset($data['file_id']) ? $data['file_id'] : null,
        ]);

        if (isset($data['new_subcategories'])) {
            foreach ($data['new_subcategories'] as $subcategory) {
                $category->subcategories()->create([
                    'name' => $subcategory,
                ]);
            }
        }

        $category->load('subcategories');

        return response()->json([
            'success' => true,
            'data' => $category,
        ], 200);
    }

    public function destroy(Category $category): JsonResponse
    {
        $deleted = $category->delete();
        $responseCode = $deleted ? 200 : 404;
        $message = $deleted ? 'Category deleted' : 'Category not found';
        $success = $deleted ? true : false;

        return response()->json(
            [
                'success' => $success,
                "message" => $message,
                'data' => $category,
            ],
            $responseCode
        );
    }
}