<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    public function create(Request $request): JsonResponse
    {
        $rules = [
            'name' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string',
            'pin' => 'nullable|string|min:4|unique:users,pin',
            'phone_number' => 'string',
            'type' => 'required|in:admin,seller,customer',
            'address' => 'nullable|string'
        ];

        $validator = \Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }

        $data = $validator->validated();

        $user = new User(
            [
                'name' => $data['name'],
                'email' => $data['email'],
                'pin' => $data['pin'] ?? null,
                'phone_number' => $data['phone_number'],
                'type' => $data['type'],
                'password' => Hash::make(
                    $data['password']
                ),
            ]
        );


        $user->save();

        if ($data['type'] == 'customer') {
            $customer = new Customer([
                'user_id' => $user->id,
                'name' => $data['name'],
                'phone_number' => $data['phone_number'],
                'address' => isset($data['address']) ? $data['address'] : null,
                'type' => 'store',
                'created_by' => $user->id
            ]);
            $customer->save();
        }

        return response()->json([
            'message' => 'Successfully created user!',
            'data' => $user,
            'token' => $user->createToken('API TOKEN')->plainTextToken
        ], 201);
    }





    public function login(Request $request): JsonResponse
    {
        $rules = [
            'email' => 'required|email',
            'password' => 'required|string',
        ];

        $validator = \Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Invalid login details',
                    'errors' => $validator->errors(),

                ],
                400
            );
        }

        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid login details'
            ], 401);
        }

        $user = User::where('email', $request->email)->firstOrFail();

        if ($user->type == 'customer') {
            $user->load('customers');
        }

        return response()->json([
            'success' => true,
            'message' => 'Successfully logged in!',
            'data' => $user,
            'token' => $user->createToken('API TOKEN')->plainTextToken
        ]);

    }

    public function loginByPin(Request $request): JsonResponse
    {
        $rules = [
            'pin' => 'required|string|min:4',
        ];

        $validator = \Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Invalid login details',
                    'errors' => $validator->errors(),

                ],
                400
            );
        }


        try {
            $user = User::where('pin', $request->pin)->firstOrFail();
        } catch (\Throwable $th) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid login details'
            ], 401);
        }

        return response()->json([
            'success' => true,
            'message' => 'Successfully logged in!',
            'data' => $user,
            'token' => $user->createToken('API TOKEN')->plainTextToken
        ]);

    }

    public function logout(): JsonResponse
    {
        // auth()->user()->tokens()->delete();
        auth()->user()->currentAccessToken()->delete();
        return response()->json([
            'success' => true,
            'message' => 'Successfully logged out!'
        ], 200);

    }

    public function checkSession(): JsonResponse
    {
        $user = auth()->user();
        $user->load('customers');
        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'Session is active!'
        ], 200);
    }

    public function updateProfileInfo(Request $request): JsonResponse
    {
        $user = auth()->user();

        $rules = [
            'email' => 'required|email|unique:users,email,' . $user->id,
            'name' => 'required|string',
            'phone_number' => 'required|string',
            'address' => 'nullable|string'
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Invalid login details',
                    'errors' => $validator->errors(),

                ],
                400
            );
        }

        $user->email = $validator->validated()['email'];
        $user->name = $validator->validated()['name'];
        $user->phone_number = $validator->validated()['phone_number'];

        if ($request->address) {
            $user->customers()->where(
                ['type' => 'store']
            )->update(
                    ['address' => $request->address]
                );
        }

        $user->save();
        $user->load('customers');
        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'User info changed successfully!'
        ], 200);
    }
    public function changeCredentials(Request $request): JsonResponse
    {
        $user = auth()->user();


        $rules = [
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:6',
        ];

        $validator = \Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Invalid login details',
                    'errors' => $validator->errors(),

                ],
                400
            );
        }

        $user->email = $validator->validated()['email'];

        if ($request->password) {
            $user->password = Hash::make($request->password);
        }

        $user->save();
        return response()->json([
            'success' => true,
            'data' => $user,
            'message' => 'User info changed successfully!'
        ], 200);
    }
}